%% RUN_AIRPERF  –  Pipeline Asignaciones 1–4 (autocontenido)
clear; clc; close all;

%% -------------------- INPUTS --------------------
inputs.S        = 992.8;     % [ft^2]
inputs.W        = 79600;     % [lbf]
inputs.AR       = 7.51;      % [-]
inputs.e        = 0.612;     % [-]
inputs.T_SL_tot = 30858;     % [lbf] empuje total a SL
inputs.h_ft     = 30000;     % [ft]
inputs.M_cap    = 0.85;      % [-]
inputs.TSFC_nom   = 0.60;    % [1/hr]
inputs.TSFC_bleed = 0.608;   % [1/hr]
inputs.use_bleed  = true;
inputs.LD_max   = 15.9;      % [-]
inputs.Wi_over_Wf= 1.10;     % [-]
inputs.M_grid   = (0.78:0.01:0.86)';

% CD0 path (puedes cambiar a curva ΔCD0 si la tienes digitalizada)
CD0_base  = 0.0055;   % const.
dCD0      = 0.0;      % incremento (0 si no hay drag-rise tabulada)

%% -------------------- CÁLCULOS --------------------
% (1) ISA
atm   = isa_std_atmos(inputs.h_ft);

% (2) Magnitudes base
base  = base_quantities(atm, inputs);

% (3) C_D0,total = C_D0,base + ΔC_D0
CD0tot = build_CD0_total(inputs.M_grid, CD0_base, dCD0);

% (4) Tabla D(M)
T     = build_drag_table(inputs.M_grid, CD0tot, base, inputs);

% (5) Chequeo @ M_cap
cap   = check_mach_cap(inputs.M_cap, CD0tot, base, inputs);

% (6) Breguet (E,R)
breg  = breguet_ER(inputs, base, atm);

%% -------------------- SALIDAS (EXPLICADAS) --------------------
printSection("Atmósfera ISA y magnitudes base (Asig. 1–2)");

explain('h_ft',  inputs.h_ft,  'ft',   'Altitud de cálculo');
explain('T_K',   atm.T_K,      'K',    'Temperatura ISA (Kelvin)');
explain('T_R',   atm.T_R,      'R',    'Temperatura ISA (Rankine)');
explain('a',     atm.a_fts,    'ft/s', 'Velocidad del sonido a = sqrt(\gamma R T)');
explain('rho',   atm.rho,      'slug/ft^3','Densidad del aire');
explain('sigma', atm.sigma,    '-',    'Relación de densidades \sigma = \rho/\rho_0');

explain('q0',    base.q0,      'lbf/ft^2','Presión dinámica a=1; q = q0 M^2');
explain('q0S',   base.q0S,     'lbf',     'Producto q0·S (útil en D y C_L(M))');
explain('k',     base.k,       '-',       'Coeficiente inducido k = 1/(\pi A R e)');
explain('TSFC',  base.TSFC,    '1/hr',    'Consumo específico usado (bleed si aplica)');

printSection("Modelo aerodinámico (Asig. 3–4)");
println("C_D(M) = C_{D0,tot}(M) + k C_L^2(M), \quad C_L(M)=W/(q_0 S M^2)");
println("D_{par}=(q_0 S) C_{D0,tot} M^2, \quad D_{ind}=(k W^2/q_0S)\,(1/M^2)");

printSection(sprintf("Tope operativo y chequeo T vs D @ M = %.2f", inputs.M_cap));
explain('C_D,cap', cap.CD_cap, '-',   'Coeficiente de arrastre a M_{cap}');
explain('D_cap',   cap.D_cap,  'lbf', 'Arrastre total a M_{cap}');
explain('T_avail', cap.T_avail,'lbf', 'Empuje disponible (T_{SL}\cdot\sigma)');
println(cap.governing);

printSection("Breguet (E, R)");
explain('V',      breg.V_kt,   'kt',  'Velocidad de crucero (V=M_{cap} a)');
explain('E',      breg.E_hr,   'h',   'Endurance: (1/TSFC)(L/D) ln(W_i/W_f)');
explain('R',      breg.R_nm,   'NM',  'Range: V · E');

%% -------------------- GRÁFICO rápido --------------------
figure; plot(T.M,T.D_par_lbf,'-o'); hold on;
plot(T.M,T.D_ind_lbf,'-s'); plot(T.M,T.D_tot_lbf,'-x'); grid on;
xlabel('Mach'); ylabel('Drag [lbf]');
legend('D_{par}','D_{ind}','D_{tot}','Location','best');
title('Desglose de arrastre vs Mach');

%% -------------------- HELPERS LOCALES --------------------
function printSection(titleStr)
    fprintf('\n=== %s ===\n', titleStr);
end

function println(s)
    fprintf('  %s\n', s);
end

function explain(symb, val, unit, txt)
    if isnumeric(val), sval = num2str(val,'%.5g'); else, sval = char(val); end
    fprintf('  %-8s = %-12s [%s]  %s\n', symb, sval, unit, txt);
end






% ============================================================
%   AIRCRAFT PERFORMANCE & DESIGN – PIPELINE ASIGNACIONES 1–4
% ============================================================
%
% 🔹 Propósito
% Este conjunto de funciones automatiza los cálculos de 
% performance (Asignaciones 1–4), permitiendo trabajar tanto
% con CD0 base constante como con la curva ΔCD0(M) digitalizada.
% Los módulos están diseñados para ejecutarse en cadena y 
% generar tablas, gráficas y resúmenes con explicación.
%
% 🔹 Flujo de ejecución principal (pipeline)
%
% run_airperf.m
%   └─> isa_std_atmos     → atm
%   └─> base_quantities   → base
%   └─> build_CD0_total   → CD0tot
%   └─> build_drag_table  → T
%   └─> check_mach_cap    → cap
%   └─> breguet_ER        → breg
%   └─> describe_airperf_outputs(atm, base, CD0tot, T, cap, breg)
%
% 🔹 Descripción de cada módulo
%
% isa_std_atmos(h_ft)
%   ENTRADA: altitud [ft]
%   SALIDA : atm struct con:
%       T_K, T_R, a_fts, rho, sigma
%   → Modelo de atmósfera estándar (ISA).
%
% base_quantities(atm, inputs)
%   ENTRADA: atm, struct inputs con {S, W, AR, e, TSFC, T_SL_tot}
%   SALIDA : base struct con:
%       q0, q0S, k, TSFC corregido
%   → Calcula magnitudes base aerodinámicas.
%
% build_CD0_total(M_grid, CD0_base, dCD0)
%   ENTRADA: malla Mach, CD0_base, curva ΔCD0(M)
%   SALIDA : CD0tot vector
%   → Construye la curva de arrastre parasitario.
%     (dos caminos: CD0_base constante o digitalización ΔCD0).
%
% build_drag_table(M_grid, CD0tot, base, inputs)
%   ENTRADA: Mach grid, CD0tot, base, inputs
%   SALIDA : tabla T con columnas:
%       M, C_D0, C_D, C_L, D_par, D_ind, D_tot
%   → Calcula arrastres parasitario, inducido y total.
%
% check_mach_cap(M_cap, CD0tot, base, inputs)
%   ENTRADA: Mach operativo
%   SALIDA : cap struct con:
%       C_D_cap, D_cap, T_avail
%   → Compara empuje disponible vs arrastre.
%     Determina si gobierna Mach límite o empuje.
%
% breguet_ER(inputs, base, atm)
%   ENTRADA: TSFC, (L/D), Wi/Wf, Mach operativo
%   SALIDA : breg struct con:
%       E (endurance), R (range)
%   → Implementa ecuaciones de Breguet.
%
% describe_airperf_outputs(atm, base, CD0tot, T, cap, breg)
%   ENTRADA: todos los structs anteriores
%   SALIDA : impresión en consola con unidades y explicación
%   → Resume y documenta resultados.
%
% 🔹 Notas de uso
% - Siempre arrancar desde run_airperf.m
% - Cada módulo puede probarse aislado llamando con sus argumentos.
% - Diseñado para Matlab Online con entrada/salida clara y
%   documentada para reportes técnicos en LaTeX.
%
% ============================================================
